<?php
/**
 * Admin Class
 *
 * Handles the Admin side functionality of plugin
 *
 * @package WP News and Scrolling Widgets Pro
 * @since 1.1.5
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Wpnw_Pro_Admin {

	function __construct() {

		// Action to add metabox
		add_action( 'add_meta_boxes', array( $this, 'wpnw_pro_news_metabox' ) );

		// Action to save metabox
		add_action( 'save_post_'.WPNW_PRO_POST_TYPE, array($this, 'wpnw_pro_save_metabox_value' ), 10, 2 );

		// Action to register admin menu
		add_action( 'admin_menu', array( $this, 'wpnw_pro_register_menu' ) );

		// Shortocde Preview
		add_action( 'current_screen', array( $this, 'wpnw_generate_preview_screen' ) );

		// Action to register plugin settings
		add_action ( 'admin_init', array( $this, 'wpnw_pro_admin_processes' ) );

		// Action to add category filter dropdown
		add_action( 'restrict_manage_posts', array( $this, 'wpnw_pro_add_post_filters' ), 50 );

		// Filter to add row data
		add_filter( 'post_row_actions', array( $this, 'wpnw_pro_add_post_row_data' ), 10, 2 );

		// Filter to add row action in category table
		add_filter( WPNW_PRO_CAT.'_row_actions', array( $this, 'wpnw_pro_add_tax_row_data' ), 10, 2 );

		// Action to add sorting link at News listing page
		add_filter( 'views_edit-'.WPNW_PRO_POST_TYPE, array( $this, 'wpnw_pro_sorting_link' ) );

		// Action to add custom column to News listing
		add_filter( 'manage_'.WPNW_PRO_POST_TYPE.'_posts_columns', array( $this, 'wpnw_pro_posts_columns' ) );

		// Action to make columns sortable at post listing page
		add_filter( 'manage_edit-'.WPNW_PRO_POST_TYPE.'_sortable_columns', array( $this, 'wpnw_pro_sortable_post_columns' ) );

		// Action to add custom column data to News listing
		add_action('manage_'.WPNW_PRO_POST_TYPE.'_posts_custom_column', array( $this, 'wpnw_pro_post_columns_data' ), 10, 2);

		// Action to add `Save Order` button
		add_action( 'restrict_manage_posts', array( $this, 'wpnw_pro_restrict_manage_posts' ) );

		// Ajax call to update option
		add_action( 'wp_ajax_wpnw_pro_update_post_order', array( $this, 'wpnw_pro_update_post_order' ) );
		
		// Filter to add plugin links
		add_filter( 'plugin_row_meta', array( $this, 'wpnw_pro_plugin_row_meta' ), 10, 2 );
	}

	/**
	 * News Post Settings Metabox
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_news_metabox() {
		add_meta_box( 'wpnw-pro-post-sett', __( 'News Settings', 'sp-news-and-widget' ), array( $this, 'wpnw_pro_news_sett_mb_content' ), WPNW_PRO_POST_TYPE, 'normal', 'high' );
	}

	/**
	 * News Post Settings Metabox HTML
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_news_sett_mb_content() {
		include_once( WPNW_PRO_DIR .'/includes/admin/metabox/wpnw-post-sett-metabox.php');
	}

	/**
	 * Function to save metabox values
	 * 
	 * @since 1.0.0
	 */
	function wpnw_pro_save_metabox_value( $post_id, $post ) {

		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )							// Check Autosave
		|| ( empty( $post_id ) || empty( $post ) )
		|| ( empty( $_POST['post_ID'] ) || absint( $_POST['post_ID'] ) !== $post_id ) )	// Check Revision
		{
			return $post_id;
		}

		$prefix = WPNW_META_PREFIX; // Taking metabox prefix

		// Taking variables
		$read_more_link = isset( $_POST[$prefix.'more_link'] ) ? wpnw_pro_clean_url( $_POST[$prefix.'more_link'] ) : '';

		$setts						= array();
		$setts['read_more_text']	= isset( $_POST[$prefix.'setts']['read_more_text'] ) ? wpnw_pro_clean_html( $_POST[$prefix.'setts']['read_more_text'] ) : '';

		update_post_meta( $post_id, $prefix.'more_link', $read_more_link );
		update_post_meta( $post_id, $prefix.'setts', $setts );
	}

	/**
	 * Function to register admin menus
	 * 
	 * @since 1.1.5
	 */
	function wpnw_pro_register_menu() {

		// Shortocde Builder
		add_submenu_page( 'edit.php?post_type='.WPNW_PRO_POST_TYPE, __( 'Shortcode Builder - WP News and Scrolling Widgets Pro', 'sp-news-and-widget' ), __( 'Shortcode Builder', 'sp-news-and-widget' ), 'edit_posts', 'wpnw-shrt-mapper', array($this, 'wpnw_shortcode_mapper_page') );

		// Settings Page
		add_submenu_page( 'edit.php?post_type='.WPNW_PRO_POST_TYPE, __( 'Settings - WP News and Scrolling Widgets Pro', 'sp-news-and-widget' ), __( 'Settings', 'sp-news-and-widget' ), 'manage_options', 'wpnw-pro-settings', array($this, 'wpnw_pro_settings_page') );

		// Shortocde Preview
		add_submenu_page( null, __( 'Shortcode Preview', 'sp-news-and-widget' ), __( 'Shortcode Preview', 'sp-news-and-widget' ), 'edit_posts', 'wpnw-preview', array($this, 'wpnw_shortcode_preview_page') );
	}

	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @since 2.1.3
	 */
	function wpnw_shortcode_mapper_page() {
		include_once( WPNW_PRO_DIR . '/includes/admin/shortcode-mapper/wpnw-shortcode-mapper.php' );
	}

	/**
	 * Function to handle the setting page html
	 * 
	 * @since 1.1.5
	 */
	function wpnw_pro_settings_page() {
		include_once( WPNW_PRO_DIR . '/includes/admin/settings/wpnw-settings.php' );
	}

	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @since 2.1.3
	 */
	function wpnw_shortcode_preview_page() {
	}

	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @since 2.1.3
	 */
	function wpnw_generate_preview_screen( $screen ) {
		if( $screen->id == 'admin_page_wpnw-preview' ) {
			include_once( WPNW_PRO_DIR . '/includes/admin/shortcode-mapper/shortcode-preview.php' );
			exit;
		}
	}

	/**
	 * Function register setings
	 * 
	 * @since 1.1.5
	 */
	function wpnw_pro_admin_processes() {
		
		// If plugin notice is dismissed
		if( isset($_GET['message']) && 'wpnw-pro-plugin-notice' == $_GET['message'] ) {
			set_transient( 'wpnw_pro_install_notice', true, 604800 );
		}

		// If plugin notice is dismissed
		if( isset($_GET['message']) && 'wpnw-pro-plugin-license-exp-notice' == $_GET['message'] ) {
			set_transient( 'wpnw_pro_license_exp_notice', true, 864000 );
		}

		// Reset default settings
		if( ! empty( $_POST['wpnw_reset_settings'] ) ) {
			
			// Default Settings
			wpnw_pro_set_default_settings();
		}

		// Register Plugin Setting
		register_setting( 'wpnw_pro_plugin_options', 'wpnw_pro_options', array( $this, 'wpnw_pro_validate_options' ) );


		// Rewrite rules after editing post type slug 
		if( ( isset($_GET['page']) && 'wpnw-pro-settings' == $_GET['page'] ) && ! empty( $_GET['settings-updated'] ) ) { 

			$wpnw_flush_rule = get_transient( 'wpnw_flush_rule' );
			
			if( $wpnw_flush_rule ) {
			
				// IMP to call to generate new rules
				flush_rewrite_rules( false );

				// Delete transient after resetting flush
				delete_transient('wpnw_flush_rule');
			}
		}
	}

	/**
	 * Validate Settings Options
	 * 
	 * @since 1.1.5
	 */
	function wpnw_pro_validate_options( $input ) {

		$post_type_slug 		= wpnw_pro_get_option( 'post_type_slug', 'news' );
		$post_archive_slug 		= wpnw_pro_get_option( 'post_archive_slug', 'news-category' );
		$disable_archive_page 	= wpnw_pro_get_option( 'disable_archive_page' );

		$input['post_guten_editor'] 	= ! empty( $input['post_guten_editor'] ) 		? 1 												: 0;
		$input['post_type_slug'] 		= ! empty( $input['post_type_slug'] ) 			? sanitize_title( $input['post_type_slug'] )		: 'news';
		$input['disable_archive_page'] 	= ! empty( $input['disable_archive_page'] ) 	? 1 												: 0;
		$input['post_archive_slug'] 	= ! empty( $input['post_archive_slug'] ) 		? sanitize_title( $input['post_archive_slug'] )		: 'news-category';
		$input['default_img'] 			= isset( $input['default_img'] ) 				? wpnw_pro_clean_url( $input['default_img'] )		: '';
		$input['custom_css'] 			= isset( $input['custom_css'] ) 				? sanitize_textarea_field( $input['custom_css'] )	: '';

		if( ( $post_type_slug != $input['post_type_slug'] ) || ( $post_archive_slug != $input['post_archive_slug'] ) || ( $disable_archive_page != $input['disable_archive_page'] ) )	{
			set_transient( 'wpnw_flush_rule', 1 );
		}

		return $input;
	}

	/**
	 * Add category dropdown to Slider listing page
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_add_post_filters() {

		global $typenow;

		if( $typenow == WPNW_PRO_POST_TYPE ) {

			$wpnw_pro_cat = isset( $_GET[WPNW_PRO_CAT] ) ? wpnw_pro_clean( $_GET[WPNW_PRO_CAT] ) : '';

			$dropdown_options = apply_filters('wpnw_pro_cat_filter_args', array(
					'show_option_none'  => __( 'All Categories', 'sp-news-and-widget' ),
					'option_none_value' => '',
					'hide_empty' 		=> 1,
					'hierarchical' 		=> 1,
					'show_count' 		=> 0,
					'orderby' 			=> 'name',
					'name'				=> WPNW_PRO_CAT,
					'taxonomy'			=> WPNW_PRO_CAT,
					'selected' 			=> $wpnw_pro_cat,
					'value_field'		=> 'slug',
				));
			wp_dropdown_categories( $dropdown_options );
		}
	}

	/**
	 * Function to add custom quick links at post listing page
	 * 
	 * @since 1.1.9
	 */
	function wpnw_pro_add_post_row_data( $actions, $post ) {

		if( $post->post_type == WPNW_PRO_POST_TYPE ) {
			return array_merge( array( 'wpos_id' => esc_html__('ID:', 'sp-news-and-widget') .' '. esc_html( $post->ID ) ), $actions );
		}

		return $actions;
	}

	/**
	 * Function to add category row action
	 * 
	 * @since 1.0
	 */
	function wpnw_pro_add_tax_row_data( $actions, $tag ) {
		return array_merge( array( 'wpos_id' => esc_html__('ID:', 'sp-news-and-widget') .' '. esc_html( $tag->term_id ) ), $actions );
	}

	/**
	 * Add 'Sort News' link at News listing page
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_sorting_link( $views ) {

		global $wpos_post_order;

		if( ! isset( $views['wpos_sort'] ) && 'wpnw' == $wpos_post_order ) {
			$class 				= ! empty( $_GET['wpnw_sort'] ) ? 'current' : '';
			$query_string 		= add_query_arg( array( 'orderby' => false, 'order' => false, 'wpnw_sort' => 1 ) );
			$views['wpos_sort'] = '<a href="' . esc_url( $query_string ) . '" class="' . esc_attr( $class ) . '">' . esc_html__( 'Sort Post', 'sp-news-and-widget' ) . '</a>';
		}

		return $views;
	}

	/**
	 * Add custom column to News listing page
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_posts_columns( $columns ){

		global $wpos_post_order;

		if( ! isset( $columns['wpos_post_order'] ) ) {

			$wpos_post_order				= 'wpnw';
			$new_columns['wpos_post_order']	= esc_html__( 'Order', 'sp-news-and-widget' );

			$columns = wpnw_pro_add_array( $columns, $new_columns, 1, true );
		}

		return $columns;
	}

	/**
	 * Make post column sortable
	 * 
	 * @since 1.4.4
	 */
	function wpnw_pro_sortable_post_columns( $columns ) {

		$columns['wpos_post_order']	= array(
												0 => 'menu_order date',
												1 => false,
											);
		return $columns;
	}

	/**
	 * Add custom column data to News listing page
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_post_columns_data( $column, $post_id ) {

		global $post, $wpos_post_order;

		if( $column == 'wpos_post_order' && 'wpnw' == $wpos_post_order ) {

			$post_menu_order = isset( $post->menu_order ) ? $post->menu_order : '';

			echo '<span class="wpnw-post-order-'.esc_attr( $post_id ).'">'. esc_html( $post_menu_order ) .'</span>';

			if ( ! empty( $_GET['wpnw_sort'] ) || ( ! empty( $_POST['action'] ) && 'inline-save' == $_POST['action'] ) ) {
				echo "<input type='hidden' value='".esc_attr( $post_id )."' name='wpnw_pro_post[]' class='wpnw-post-order-inp' id='wpnw-post-order-inp-".esc_attr( $post_id )."' />";
			}
		}
	}

	/**
	 * Add Save button to News listing page
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_restrict_manage_posts(){

		global $typenow;

		if( $typenow == WPNW_PRO_POST_TYPE && ! empty( $_GET['wpnw_sort'] ) ) {

			echo "<span class='spinner wpnw-spinner'></span>
				  <input type='button' name='wpnw_save_order' class='button button-secondary right wpnw-save-order' id='wpnw-save-order' value='".esc_html__( 'Save Sort Order', 'ticker-ultimate' )."' data-nonce='". esc_attr( wp_create_nonce( 'wpnw-save-post-order' ) )."' />";
		}
	}

	/**
	 * Update News order
	 * 
	 * @since 1.1.6
	 */
	function wpnw_pro_update_post_order() {

		// Taking some defaults
		$result 			= array();
		$result['success'] 	= 0;
		$result['msg'] 		= esc_js( __( 'Sorry, Something happened wrong.', 'ticker-ultimate') );
		$wpnw_post_nonce	= isset( $_POST['nonce'] ) ? wpnw_pro_clean( $_POST['nonce'] ) : '';

		if( ! empty( $_POST['form_data'] ) && wp_verify_nonce( $wpnw_post_nonce, 'wpnw-save-post-order' ) ) {

			$user_id		= get_current_user_id();
			$form_data 		= parse_str( $_POST['form_data'], $output_arr );
			$post_type		= ! empty( $output_arr['post_type'] )	? $output_arr['post_type']		: '';
			$wpnw_posts 	= !empty($output_arr['wpnw_pro_post'])	? $output_arr['wpnw_pro_post']	: '';
			$paged			= ! empty( $output_arr['paged'] )		? (int) $output_arr['paged']	: 1;
			$paged			= ( $paged - 1 );

			if( ! empty( $wpnw_posts ) && ! empty( $post_type ) ) {

				$per_page = (int) get_user_option( "edit_{$post_type}_per_page" );

				if ( empty( $per_page ) || $per_page < 1 ) {
					$per_page = 20;
				}

				$post_menu_order = (int) ( $paged * $per_page );

				// Loop od ids
				foreach ( $wpnw_posts as $wpbab_post_key => $wpnw_post ) {
					
					// Update post order
					$update_post = array(
						'ID' 			=> $wpnw_post,
						'menu_order'	=> $post_menu_order,
					);

					// Update the post into the database
					wp_update_post( $update_post );

					// Taking all post record
					$result['posts'][ $wpnw_post ] = $post_menu_order;

					$post_menu_order++;
				}

				$result['success'] 	= 1;
				$result['msg'] 		= esc_js( __( 'Post order saved successfully.', 'sp-news-and-widget' ) );
			}
		}

		wp_send_json( $result );
	}

	/**
	 * Function to unique number value
	 * 
	 * @since 1.1.5
	 */
	function wpnw_pro_plugin_row_meta( $links, $file ) {

		if ( $file == WPNW_PRO_PLUGIN_BASENAME ) {

			$row_meta = array(
				'docs'    => '<a href="' . esc_url( 'https://docs.essentialplugin.com/wp-news-and-scrolling-widgets-pro/?utm_source=news_post_pro&utm_medium=plugin_list&utm_campaign=plugin_quick_link' ) . '" title="' . esc_attr__( 'View Documentation', 'sp-news-and-widget' ) . '" target="_blank">' . esc_html__( 'Docs', 'sp-news-and-widget' ) . '</a>',
				'support' => '<a href="' . esc_url( 'https://www.wponlinesupport.com/wordpress-services/?utm_source=news_post_pro&utm_medium=plugin_list&utm_campaign=plugin_quick_link' ) . '" title="' . esc_attr__( 'Premium Support - For any Customization', 'sp-news-and-widget' ) . '" target="_blank">' . esc_html__( 'Premium Support', 'sp-news-and-widget' ) . '</a>',
			);
			return array_merge( $links, $row_meta );
		}
		return (array)$links;
	}
}

$wpnw_pro_admin = new Wpnw_Pro_Admin();